#!/bin/bash

###########################################################
## Copyright (c) 2002-2025 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## Log roller for crwdaq: roll (rotate) crwdaq log files ##
## located in crwdaq log directory $CRW_DAQ_VAR_TMP_DIR. ##
## Notes: this script work properly only as crwdaq child ##
## process and only while parent crwdaq process running. ##
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

readonly crwpid="$CRW_DAQ_SYS_EXE_PID"; # PID of parent crwdaq
readonly logdir="$CRW_DAQ_VAR_TMP_DIR"; # crwdaq log directory
readonly log="$logdir/$scriptbase.log"; # script log file name
readonly defloglimit="32MB";            # $1 log size limit
readonly defnmaxkeep="9";               # $2 n max keep
readonly defcompress="gz";              # $3 compressor
readonly arguments="$*";                # all arguments

#########################################
# check command $1 with PID $2 is running
#########################################
function cmd_pid_running(){
 if [ $# -ne 2 ]; then return 1; fi;
 if [ -n "$(pgrep -x $1 | grep "\b$2\b")" ]; then return 0; fi;
 return 1;
};

#################
# add line to log
#################
function add_log(){
 local prog="$CRW_DAQ_SYS_TITLE";
 local time="$(date +%Y.%m.%d-%H:%M:%S)";
 echo "$time $prog: $*" |  tee -a "$log";
};

#######################
# check/create log file
#######################
function check_log(){
 touch "$log";
 add_log "$scriptname $arguments";
};

############################################
# calculate size with KB,MB,GB,TB substitute
# use KiloByte=1024 as power of 2 (not 1000)
############################################
function calc_size(){
 local kb=1024;                     # kilobyte factor
 local args="$*";                   # fetch arguments
 args="${args,,}";                  # makes lowercase
 args="${args/b/}";                 # remove 'b' char
 args="${args/k/*$kb}";             # apply kilobytes
 args="${args/m/*$kb*$kb}";         # apply megabytes
 args="${args/g/*$kb*$kb*$kb}";     # apply gigabytes
 args="${args/t/*$kb*$kb*$kb*$kb}"; # apply terabytes
 local -i value=0;                  # result value:
 let "value=$args";                 # calculate
 echo "$value";                     # and print
};

#####################################
# example:
# find_files_over_size . '*.log' 32MB
#####################################
function find_files_over_size(){
 local dir="$1"; local name="$2";
 local -i size=$(calc_size "$3"); if [[ $size -le 0 ]]; then return; fi;
 find "$dir" -maxdepth 1 -type f -and -name "$name" -and '(' -size ${size}c -or -size +${size}c ')';
};

######
# Main
######
function main(){
 if [[ -z "$crwpid" ]]; then fatal 1 "Error: crwdaq environ (PID)."; fi;
 if [[ -z "$logdir" ]]; then fatal 1 "Error: crwdaq environ (dir)."; fi;
 if [[ -d "$logdir" ]]; then
  if cmd_pid_running crwdaq $crwpid; then
   check_log; local -i numroll=0;
   local loglimit="${1:-$defloglimit}";
   local nmaxkeep="${2:-$defnmaxkeep}";
   local compress="${3:-$defcompress}";
   loglimit="$(calc_size "$loglimit")";
   if pushd "$logdir" > /dev/null; then
    local logs="$(find_files_over_size . '*.log' "$loglimit" | xargs)";
    if [[ -n "$logs" ]]; then
     add_log "SizeLimit: $loglimit byte(s)";
     add_log "Directory: $logdir";
     add_log "RollFiles: $logs";
     for file in $logs; do
      if unix logroll -t $file $nmaxkeep $compress; then
       let numroll+=1;
      fi;
     done;
     add_log "NumRolled: $numroll";
    fi;
   fi;
   popd > /dev/null;
  else
   fatal 1 "Error: crwdaq PID $crwpid is not running.";
  fi;
 else
  fatal 1 "Error: missed crwdaq log directory.";
 fi;
};

main "$@";

##############
## END OF FILE
##############
