#!/bin/bash

###########################################################
## Copyright (c) 2002-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## Configuration Generator for VTRD devices.
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

#################################
### VTRD devices config generator
#################################

######################
# initialize variables
######################
readonly hdwcfg="${scriptbase}_hdw.cfg"; # hardware   configuration
readonly simcfg="${scriptbase}_sim.cfg"; # simulator  configuration
readonly diccfg="${scriptbase}_dic.cfg"; # DIM client configuration
readonly discfg="${scriptbase}_dis.cfg"; # DIM server configuration
readonly impcfg="${scriptbase}_imp.cfg"; # DIM import configuration
readonly devmsg="devPostMsg";            # Method to send/post messages
readonly tabcrc="../circuits/${scriptbase}_link.crc";
declare -i numhdwcfg=0;                  # Flag to include hardware config
declare -i CTRL_TOP=0;                   # Top  of main control window
declare -i CTRL_LEFT=600;                # Left of main control window
declare -i CTRL_TOP_STEP=150;
declare -i MAIN_HEIGHT=160;
declare -i MAIN_H_STEP=50;
declare -i TABCRC_TOP=0;
declare -i TABCRC_STEP=35;
declare fileprefix="";
declare mainprefix="VMCD";
declare FACN="";                         # Facility name
declare IDEN="";                         # Device identifier name
declare FACP="";                         # Facility DIM  path
declare FACF="";                         # Facility file name
declare FACL="";                         # Facility name (lower case)
declare IDNL="";                         # Device identifier name (lower case)
declare IDNF="";                         # Device identifier name (lower case)

function call(){
 echo "$*" 1>&2;
 eval "$@";
};

function rem(){
 return 0;
};

function vtrd_devices(){
 call DeleteFiles $hdwcfg $simcfg $tabcrc $diccfg $discfg;
 rem  Device Facility Identitier Port UnitId Polling;
 call VTRD   VTRD     PT1        4    21     10;
 call VTRD   VTRD     PT2        4    22     10;
 call VTRD   VTRD     PT3        4    23     10;
 call VTRD   VTRD     PT4        4    24     10;
 call VTRD   VTRD     PT5        4    25     10;
 call VTRD   VTRD     PT6        4    26     10;
 call VTRD   VTRD     PT7        4    27     10;
 cat $hdwcfg $simcfg $diccfg $discfg;
};

function VTRD(){
 call VTRD_SET $*;
 call VTRD_HDW $* >> $hdwcfg;
 call VTRD_SIM $* >> $simcfg;
 call VTRD_DIC $* >> $diccfg;
 call VTRD_DIS $* >> $discfg;
 call VTRD_TABCRC >> $tabcrc;
let "CTRL_TOP+=CTRL_TOP_STEP";
};

function VTRD_SET(){
 FACN="$1"          ; shift; # FACN - Facility name
 IDEN="$1"          ; shift; # IDEN - Identifier
 ModbusPort="$1"    ; shift; # Modbus Port number
 ModbusUnitId="$1"  ; shift; # Modbus Unit Id
 ModbusPolling="$1" ; shift; # Modbus Polling period, ms
 FACP="$(echo "$FACN" | tr '.' '/')"; ## Facility DIM  path, i.e. FACN name with . to / replacement
 FACF="$(echo "$FACN" | tr '.' '_')"; ## Facility file name, i.e. FACN name with . to _ replacement
 FACL="$(echo "$FACN" | tr '[:upper:]' '[:lower:]')"; ## Facility name to lower case
 FACF="$(echo "$FACF" | tr '[:upper:]' '[:lower:]')"; ## Facility file name to lower case
 IDNF="$(echo "$IDEN" | tr '[:upper:]' '[:lower:]')";
 IDNL="$(echo "$IDEN" | tr '[:upper:]' '[:lower:]')";
};

function VTRD_HDW(){
 echo ";***********************";
 echo "; VTRD driver - hardware";
 echo ";***********************";
 echo "[DeviceList]";
 echo "&$FACN.$IDEN.DRV = device software program";
 echo "[&$FACN.$IDEN.DRV]";
 echo "Comment = Modbus RTU Driver for VTRD device.";
 echo "InquiryPeriod  = 0";
 echo "DevicePolling  = 10, tpHigher";
 echo "ProgramSource  = ../daqpas/${fileprefix}vtrd_drv.pas";
 echo "DigitalFifo    = 1024";
 echo "AnalogFifo     = 1024";
 echo "DebugFlags     = 3";
 echo "OpenConsole    = 2";
 echo "Simulator      = 0";
 echo "ModbusPort     = $ModbusPort";
 echo "ModbusUnitId   = $ModbusUnitId";
 echo "ModbusTimeout  = 250";
 echo "ModbusDeadline = 60000";
 echo "ModbusPolling  = $ModbusPolling";
 echo "DelayOnStart   = 1000";
 echo "tagPrefix      = $FACN.$IDEN";
 echo "EncryptMethod  = RC6";
 echo "TrustedUsers   = [TrustedUsers]";
 echo "AnalogOutputs  = 1";
 echo "Link AnalogOutput 0 with curve $FACN.$IDEN.PRESSURE history 2000";
 echo "DigitalOutputs = 8";
 echo "Link DigitalOutput 0 with curve $FACN.$IDEN.STATE.INDIC history 200";
 echo "Link DigitalOutput 1 with curve $FACN.$IDEN.POLLRATE.RX history 200";
 echo "Link DigitalOutput 2 with curve $FACN.$IDEN.POLLRATE.TX history 200";
 echo "Link DigitalOutput 3 with curve $FACN.$IDEN.POLLRATE.EX history 200";
 echo "Link DigitalOutput 4 with curve $FACN.$IDEN.ERROR.COUNT history 200";
 echo "Link DigitalOutput 5 with curve $FACN.$IDEN.POLLSUMM.RX history 200";
 echo "Link DigitalOutput 6 with curve $FACN.$IDEN.POLLSUMM.TX history 200";
 echo "Link DigitalOutput 7 with curve $FACN.$IDEN.POLLSUMM.EX history 200";
 echo ";------ Save custom parameters to INI file";
 echo "CustomIniTagList  = [$FACN.$IDEN.CustomParameters.TagList]";
 echo "CustomIniFileRef  = ../data/${FACF}_${IDNF}_custom.ini";
 echo "CustomIniSection  = [$FACN.$IDEN.CustomParameters]";
 echo "CustomIniBackups  = ../../${FACF}_data/$FACF/custom";
 echo "CustomIniAutoLoad = 1";
 echo "CustomIniAutoSave = 0";
 echo "[]";
 echo "[ConfigFileList]";
 echo "ConfigFile = ../data/${FACF}_${IDNF}_custom.ini";
 echo "[]";
 echo "";
 echo "[$FACN.$IDEN.CustomParameters.TagList]";
 echo "TagList = $FACN.$IDEN.ENABLE";
 echo "[]";
 echo "";
 echo "[TagList]";
 echo "$FACN.$IDEN.DIMGUICLICK = String  ? ; The DIM click sensor data";
 echo "$FACN.$IDEN.ENABLE      = Integer 1 ; Enable polling flag";
 echo "$FACN.$IDEN.PRESSURE    = Real    0 ; Measurement pressure, Pascal";
 echo "$FACN.$IDEN.STATE       = Integer 1 ; Status indicator";
 echo "$FACN.$IDEN.ADDRESS     = Integer 0 ; Device address";
 echo "$FACN.$IDEN.BAUDRATE    = Integer 0 ; Device baudrate";
 echo "$FACN.$IDEN.POLLRATE.RX = Real    0 ; Poll rate, receiver    (per sec)";
 echo "$FACN.$IDEN.POLLRATE.TX = Real    0 ; Poll rate, transmitter (per sec)";
 echo "$FACN.$IDEN.POLLRATE.EX = Real    0 ; Poll rate, errors      (per sec)";
 echo "$FACN.$IDEN.ERROR.COUNT = Real    0 ; Error counter          (counter)";
 echo "$FACN.$IDEN.POLLSUMM.RX = Real    0 ; Poll summ, receiver    (counter)";
 echo "$FACN.$IDEN.POLLSUMM.TX = Real    0 ; Poll summ, transmitter (counter)";
 echo "$FACN.$IDEN.POLLSUMM.EX = Real    0 ; Poll summ, errors      (counter)";
 echo "[]";
 echo "";
 echo "[DataStorage]";
 echo "$FACN.$IDEN.PRESSURE    = Curve 0 10 Black 15 1 ; Measurement pressure, Pascal";
 echo "$FACN.$IDEN.STATE.INDIC = Curve 0 10 Black 15 1 ; Status indicator";
 echo "$FACN.$IDEN.POLLRATE.RX = Curve 0 10 Black 15 1 ; Poll rate, receiver    (per sec)";
 echo "$FACN.$IDEN.POLLRATE.TX = Curve 0 10 Black 15 1 ; Poll rate, transmitter (per sec)";
 echo "$FACN.$IDEN.POLLRATE.EX = Curve 0 10 Black 15 1 ; Poll rate, errors      (per sec)";
 echo "$FACN.$IDEN.ERROR.COUNT = Curve 0 10 Black 15 1 ; Error counter          (counter)";
 echo "$FACN.$IDEN.POLLSUMM.RX = Curve 0 10 Black 15 1 ; Poll summ, receiver    (counter)";
 echo "$FACN.$IDEN.POLLSUMM.TX = Curve 0 10 Black 15 1 ; Poll summ, transmitter (counter)";
 echo "$FACN.$IDEN.POLLSUMM.EX = Curve 0 10 Black 15 1 ; Poll summ, errors      (counter)";
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "$FACN.$IDEN.PARAM.PLOT = Curve_Window";
 echo "[$FACN.$IDEN.PARAM.PLOT]";
 echo "AxisX = ^R{Часов}___^N^CВремя, 0, 0.25";
 echo "AxisY = ^CСостояние_каналов^N^L___{Статус}, 0, 1000";
 echo "CurveList = $FACN.$IDEN.PRESSURE";
 echo "CurveList = $FACN.$IDEN.POLLRATE.RX";
 echo "CurveList = $FACN.$IDEN.POLLRATE.TX";
 echo "CurveList = $FACN.$IDEN.POLLRATE.EX";
 echo "CurveList = $FACN.$IDEN.STATE.INDIC";
 echo "CurveList = $FACN.$IDEN.ERROR.COUNT";
 echo "CurveList = $FACN.$IDEN.POLLSUMM.RX";
 echo "CurveList = $FACN.$IDEN.POLLSUMM.TX";
 echo "CurveList = $FACN.$IDEN.POLLSUMM.EX";
 echo "[]";
 echo "";
 echo "[&CronSrv.StartupScript]";
 echo "@cron.tab $FACN.$IDEN.PARAM.PLOT.HOME 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HOME @WinShow $FACN.$IDEN.PARAM.PLOT";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HOME @WinDraw $FACN.$IDEN.PARAM.PLOT|Top=317|Left=167|Width=1070|Height=483";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HOME @WinDraw $FACN.$IDEN.PARAM.PLOT|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar";
 echo "@cron.tab $FACN.$IDEN.PARAM.PLOT.SHOW 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.SHOW @WinShow $FACN.$IDEN.PARAM.PLOT";
 echo "@cron.tab $FACN.$IDEN.PARAM.PLOT.HIDE 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HIDE @WinHide $FACN.$IDEN.PARAM.PLOT";
 echo "@cron.tab $FACN.$IDEN.PARAM.PLOT.SELECT 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.SELECT @WinSelect $FACN.$IDEN.PARAM.PLOT";
 echo "@cron.job $FACN.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.PARAM.PLOT.HOME";
 echo "@cron.job $FACN.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.PARAM.PLOT.HIDE";
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "$FACN.$IDEN.CTRL = Circuit_Window";
 echo "[$FACN.$IDEN.CTRL]";
 echo "Circuit = ../circuits/${fileprefix}vtrd_ctrl.crc";
 echo "Link sensor VTRD.ENABLE      with tag   $FACN.$IDEN.ENABLE      device &$FACN.$IDEN.DRV";
 echo "Link sensor VTRD.STATE       with tag   $FACN.$IDEN.STATE       device &$FACN.$IDEN.DRV";
 echo "Link sensor VTRD.PRESSURE    with curve $FACN.$IDEN.PRESSURE    device &$FACN.$IDEN.DRV";
 echo "Link sensor VTRD.TOOLS       with                            device &$FACN.$IDEN.DRV";
 echo "Link sensor VTRD.POLLRATE.RX with curve $FACN.$IDEN.POLLRATE.RX device &$FACN.$IDEN.DRV";
 echo "Link sensor VTRD.POLLRATE.TX with curve $FACN.$IDEN.POLLRATE.TX device &$FACN.$IDEN.DRV";
 echo "Link sensor VTRD.POLLRATE.EX with curve $FACN.$IDEN.POLLRATE.EX device &$FACN.$IDEN.DRV";
 echo "Link sensor VTRD.POLLSUMM.EX with curve $FACN.$IDEN.POLLSUMM.EX device &$FACN.$IDEN.DRV";
 echo "[]";
 echo "";
 echo "[&CronSrv.StartupScript]";
 echo "@cron.tab $FACN.$IDEN.CTRL.HOME 0 0 0";
 echo "@cron.job $FACN.$IDEN.CTRL.HOME @WinShow $FACN.$IDEN.CTRL";
 echo "@cron.job $FACN.$IDEN.CTRL.HOME @WinDraw $FACN.$IDEN.CTRL|Left=$CTRL_LEFT|Top=$CTRL_TOP|Width=500|Height=150";
 echo "@cron.job $FACN.$IDEN.CTRL.HOME @WinDraw $FACN.$IDEN.CTRL|Options=-Min,-Max,+Close,-VScroll,-HScroll,+StatusBar";
 echo "@cron.job $FACN.$IDEN.CTRL.HOME @WinDraw $FACN.$IDEN.CTRL|Options=+Top,+Left,-Width,-Height";
 echo "@cron.tab $FACN.$IDEN.CTRL.SHOW 0 0 0";
 echo "@cron.job $FACN.$IDEN.CTRL.SHOW @WinShow $FACN.$IDEN.CTRL";
 echo "@cron.tab $FACN.$IDEN.CTRL.HIDE 0 0 0";
 echo "@cron.job $FACN.$IDEN.CTRL.HIDE @WinHide $FACN.$IDEN.CTRL";
 echo "@cron.tab $FACN.$IDEN.CTRL.SELECT 0 0 0";
 echo "@cron.job $FACN.$IDEN.CTRL.SELECT @WinSelect $FACN.$IDEN.CTRL";
 echo "@cron.job $FACN.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.CTRL.HOME";
 echo "@cron.job $FACN.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.CTRL.HIDE";
 echo "[]";
 echo "";
 echo "[$FACN.MAIN.CTRL]";
 echo "Link sensor @cron+@cron.run+$FACN.$IDEN.CTRL.SELECT with device &$FACN.MAIN.CTRL";
 echo "Link sensor $FACN.$IDEN.STATE    with tag   $FACN.$IDEN.STATE    device &$FACN.$IDEN.DRV";
 echo "Link sensor $FACN.$IDEN.PRESSURE with curve $FACN.$IDEN.PRESSURE device &$FACN.$IDEN.DRV";
 echo "[]";
 echo "";
 echo "[&DatSrv]";
 echo "CurveList = $FACN.$IDEN.PRESSURE";
 echo "CurveList = $FACN.$IDEN.POLLRATE.RX";
 echo "CurveList = $FACN.$IDEN.POLLRATE.TX";
 echo "CurveList = $FACN.$IDEN.POLLRATE.EX";
 echo "CurveList = $FACN.$IDEN.STATE.INDIC";
 echo "CurveList = $FACN.$IDEN.ERROR.COUNT";
 echo "CurveList = $FACN.$IDEN.POLLSUMM.RX";
 echo "CurveList = $FACN.$IDEN.POLLSUMM.TX";
 echo "CurveList = $FACN.$IDEN.POLLSUMM.EX";
 echo "[]";
 echo "";
};

function VTRD_SIM(){
 echo "[&$FACN.$IDEN.DRV]";
 echo "Simulator = 1";
 echo "[]";
 rem Include hardware config if one not done yet;
 if [[ "$numhdwcfg" == "0" ]]; then let "numhdwcfg=numhdwcfg+1"; else return 0; fi;
 echo "[ConfigFileList]";
 echo "ConfigFile = $hdwcfg";
 echo "[]";
};

function VTRD_TABCRC(){
 let "YPOS=TABCRC_TOP+40";
 echo "[SensorList]";
 echo "Sensor = @cron+@cron.run+$FACN.$IDEN.CTRL.SELECT";
 echo "[@cron+@cron.run+$FACN.$IDEN.CTRL.SELECT]";
 echo "Pos = 10, $YPOS";
 echo "Painter(v) = (glBorderBevel=2)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Style:[Bold]\Size:14";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_21_24_lightsteelblue1_ptmono.bmp $FACN.$IDEN";
 echo "Hint = $FACN.$IDEN.DRV - Открыть окно управления";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.PRESSURE";
 echo "[$FACN.$IDEN.PRESSURE]";
 echo "Pos = 150,  $YPOS";
 echo "LED = 8, 4, 0, %8.4E, Name:PT_Mono\Style:[Bold]\Size:14";
 echo "Painter(v) = (glBorderBevel=2)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_22_24_aqua_ptmono.bmp";
 echo "Hint = $FACN.$IDEN.PRESSURE - Измеренное давление, Па";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.STATE";
 echo "[$FACN.$IDEN.STATE]";
 echo "Pos = 290,  $YPOS";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Style:[Bold]\Size:10";
 echo "Painter(v) = (glBorderBevel=2)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_64_32_24_lime.bmp   OK";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_64_32_24_yellow.bmp TIMEOUT";
 echo "Tag#3 = 2, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_64_32_24_red.bmp    REFUSE";
 echo "Tag#4 = 3, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_64_32_24_lime.bmp   SIMULAT";
 echo "Tag#5 = 4, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_64_32_24_silver.bmp DISABLE";
 echo "Hint = $FACN.$IDEN.STATE - Состояние датчика";
 echo "[]";
 echo "";
 let "TABCRC_TOP+=TABCRC_STEP";
};

function VTRD_DIS(){
 unix dimcfg \
  -n section "[&$FACN.$IDEN.DRV]" \
  -n print    DimServerMode = 1 \
  -n end \
  -n dis_cmnd   $FACP/$IDEN/DIMGUICLICK \
  -n tag        $FACN.$IDEN.DIMGUICLICK \
  -n $devmsg "&$FACN.$IDEN.DRV @DIMGUICLICK=%**" \
  -n end \
  -n dic_cmnd   $FACP/$IDEN/DIMGUICLICK \
  -n tag        $FACN.$IDEN.DIMGUICLICK \
  -n end \
  -n dis_info   $FACP/$IDEN/ENABLE \
  -n tag        $FACN.$IDEN.ENABLE \
  -n end \
  -n dis_info   $FACP/$IDEN/PRESSURE \
  -n tag        $FACN.$IDEN.PRESSURE \
  -n end \
  -n dis_info   $FACP/$IDEN/STATE \
  -n tag        $FACN.$IDEN.STATE \
  -n end \
  -n dis_info   $FACP/$IDEN/ADDRESS \
  -n tag        $FACN.$IDEN.ADDRESS \
  -n end \
  -n dis_info   $FACP/$IDEN/BAUDRATE \
  -n tag        $FACN.$IDEN.BAUDRATE \
  -n end \
  -n dis_info   $FACP/$IDEN/POLLRATE/RX+TX+EX \
  -n tag        $FACN.$IDEN.POLLRATE.RX \
  -n tag        $FACN.$IDEN.POLLRATE.TX \
  -n tag        $FACN.$IDEN.POLLRATE.EX \
  -n end \
  -n dis_info   $FACP/$IDEN/POLLSUMM/RX+TX+EX \
  -n tag        $FACN.$IDEN.POLLSUMM.RX \
  -n tag        $FACN.$IDEN.POLLSUMM.TX \
  -n tag        $FACN.$IDEN.POLLSUMM.EX \
  -n end \
  -n dis_info   $FACP/$IDEN/ERROR/COUNT \
  -n tag        $FACN.$IDEN.ERROR.COUNT \
  -n end \
  -n ;
};

function VTRD_DIC(){
 unix dimcfg \
  -n section "[&$FACN.$IDEN.DRV]" \
  -n print    DimClientMode = 1 \
  -n end \
  -n dic_cmnd   $FACP/$IDEN/DIMGUICLICK \
  -n tag        $FACN.$IDEN.DIMGUICLICK \
  -n end \
  -n dic_info   $FACP/$IDEN/ENABLE \
  -n tag        $FACN.$IDEN.ENABLE \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLL.ENABLE" \
  -n end \
  -n dic_info   $FACP/$IDEN/PRESSURE \
  -n tag        $FACN.$IDEN.PRESSURE \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.PRESSURE" \
  -n end \
  -n dic_info   $FACP/$IDEN/STATE \
  -n tag        $FACN.$IDEN.STATE \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.STATE" \
  -n end \
  -n dic_info   $FACP/$IDEN/ADDRESS \
  -n tag        $FACN.$IDEN.ADDRESS \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.ADDRESS" \
  -n end \
  -n dic_info   $FACP/$IDEN/BAUDRATE \
  -n tag        $FACN.$IDEN.BAUDRATE \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.BAUDRATE" \
  -n end \
  -n dic_info   $FACP/$IDEN/POLLRATE/RX+TX+EX \
  -n tag        $FACN.$IDEN.POLLRATE.RX \
  -n tag        $FACN.$IDEN.POLLRATE.TX \
  -n tag        $FACN.$IDEN.POLLRATE.EX \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLLRATE.RX" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLLRATE.TX" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLLRATE.EX" \
  -n end \
  -n dic_info   $FACP/$IDEN/POLLSUMM/RX+TX+EX \
  -n tag        $FACN.$IDEN.POLLSUMM.RX \
  -n tag        $FACN.$IDEN.POLLSUMM.TX \
  -n tag        $FACN.$IDEN.POLLSUMM.EX \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLLSUMM.RX" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLLSUMM.TX" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLLSUMM.EX" \
  -n end \
  -n dic_info   $FACP/$IDEN/ERROR/COUNT \
  -n tag        $FACN.$IDEN.ERROR.COUNT \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.ERROR.COUNT" \
  -n end \
  -n ;
};

##############
# delete files
##############
function DeleteFiles(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then rm -f "$1"; fi;
  shift;
 done;
};

################
# main function.
################
function main(){
 if pushd $scriptHOME >/dev/null 2>&1; then
  vtrd_devices;
 fi;
 popd >/dev/null 2>&1;
 local code=$?;
 if [ $code -eq 0 ]; then
  local msg="$(langstr ru "конфигурация успешно создана." en "configuration created.")";
  unix tooltip-notifier preset stdOk delay 7000 text "$scriptname: $msg";
 else
  local msg="$(langstr ru "ошибка создания конфигурации." en "failed on create configuration.")";
  fatal $code "$scriptname: $msg";
 fi;
 return $code;
};

main "$@";

##############
## END OF FILE
##############
