#!/bin/bash

function about(){
 echo "$(basename ${BASH_SOURCE[0]}) version 20250412";
 echo "##############################################################";
 echo "# Shell Installation Script.                                 #";
 echo "# Copyright (c) 2023-2024 Alexey Kuryakin daqgroup@mail.ru   #";
 echo "##############################################################";
};

function note(){ return; };
function fatal(){ echo -ne "\n$1\n\n"; exit 1; };

note "Script identification.";
note "######################";
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

note "Extract target package name from script name.";
note "#############################################";
readonly targetname="$(echo "$scriptbase" | sed 's/^[a-z]*-//')";

note "The payload directory contains user installer + data archive(s).";
note "################################################################";
readonly payloaddir="$scriptHOME";

note "Include the install-library.sh library source.";
note "##############################################";
readonly installlib="$payloaddir/install-library.sh";
source $installlib || fatal "Error: not found $installlib.";

note "#######################";
note "Install target package.";
note "#######################";
function install_target(){
 echo "";
 echo "Install: $targetname";
 echo "";
 common_predinst;
 note "##################################";
 note "### Begin User Installer code: ###";
 note "##################################";
 
 note "########################################";
 note "Check required packages to be installed.";
 note "########################################";
 check_install_dpkg zenity makeself zip unzip xz-utils p7zip;
 check_install_dpkg gnuplot busybox;
 note "#############################";
 note "Unpack archive to target dir.";
 note "#############################";
 local archive="$targetname.tar";
 local targdir="/opt/daqgroup/suite";
 check_confirm_iam_user || fatal "Cancel: user mode expected.";
 tar_list_check_exist $archive $targdir || fatal "Cancel: file already exist.";
 root_mkdir $targdir;
 sudo rm -fvr $targdir/crwdaq/*;
 unpack_tar $archive $targdir 777 root;
 note "Make all files readable.";
 sudo chmod -c ugo+rx $targdir;
 sudo chmod -c ugo+rx $targdir/..;
 sudo chmod -cR ugo+r $targdir/*;
 #note "Set Owner for all files.";
 #sudo chown -cR $USER:$USER $targdir/*;
 if [ -e $targdir/crwdaq/ ]; then
  if [ -L /opt/crwdaq ]; then
   sudo rm -fv /opt/crwdaq;
  fi;
  sudo ln -sv -T $targdir/crwdaq /opt/crwdaq;
 fi;
 note "Make some files & directories writeble";
 sudo chmod -c  ugo+rw  $targdir/crwdaq/resource/manual/crw-daq-colors.htm;
 sudo chmod -c  ugo+rwx $(find $targdir/crwdaq/demo -type d | xargs);
 sudo chmod -cR ugo+rw  $targdir/crwdaq/demo/*;
 sudo chmod -c  ugo+rwx $targdir/crwdaq/resource/daqsite/stdlib/bitmaps;
 note "Set special capabilities.";
 local wanted_caps="cap_net_bind_service,cap_sys_nice+ep";
 sudo setcap $wanted_caps $targdir/crwdaq/crwdaq;
 sudo getcap $targdir/crwdaq/crwdaq;
 note "Update xterm configuration.";
 if unix which xterm-config >/dev/null 2>&1; then unix xterm-config -u -a; fi;
 note "Set System Start Menu items.";
 $targdir/crwdaq/resource/launch/setsystemmenu.sh;
 note "Update dunst and make it single nofifier.";
 if which dunst >/dev/null 2>&1; then
  sudo crwkit dunstcfg -u -s;
  crwkit dunstcfg -u -s;
 else
  echo "Skip dunst - not installed yet.";
 fi;

 note "##################################";
 note "### End of User Installer code ###";
 note "##################################";
 common_postinst;
};

note "########";
note "# Main #";
note "########";
function main(){
 about; print_identification; sleep 1;
 check_sudo_is_avail || fatal "Error: sudo is not available.";
 check_install_package zenity makeself zip unzip xz:xz-utils 7za:p7zip;
 local ask="$(zen_hello_user)\n\n$(zen_do_you_want_to $(langstr en 'Run' ru 'Запустить'))\n";
 ask="$ask\n<big><b><span fgcolor='green'>install-$targetname.sh</span></b>?</big>";
 if ask_zenity 0 --question --text "$ask" --timeout 180 >/dev/null 2>&1; then
  install_target;
  echo -ne "\nDone\n\n";
 else
  echo -ne "\nCancel\n\n";
 fi;
};

mkdir -p $HOME/.local/share/makeself;

main "$@" | tee $HOME/.local/share/makeself/$scriptbase.log;

#############
# END OF FILE
#############
